<?php

class PayPalClient {

    private $cid;
    private $sec;
    private $base;

    public function __construct($cid, $sec, $sandbox = true)
    {
        $this->cid  = $cid;
        $this->sec  = $sec;
        $this->base = $sandbox
            ? 'https://api-m.sandbox.paypal.com'
            : 'https://api-m.paypal.com';
    }

    private function req($method, $endpoint, $body = null, $token = null)
    {
        $ch = curl_init($this->base . $endpoint);

        $headers = ['Content-Type: application/json'];
        if ($token) {
            $headers[] = 'Authorization: Bearer ' . $token;
        }

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST  => $method,
            CURLOPT_HTTPHEADER     => $headers
        ]);

        if ($body !== null) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($body));
        }

        $raw = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($raw === false) {
            $e = curl_error($ch);
            curl_close($ch);
            throw new Exception($e);
        }

        curl_close($ch);

        $json = json_decode($raw, true);

        if ($status < 200 || $status >= 300) {
            throw new Exception("PayPal HTTP $status: $raw");
        }

        return $json;
    }

    public function token()
    {
        $ch = curl_init($this->base . '/v1/oauth2/token');

        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_USERPWD        => $this->cid . ':' . $this->sec,
            CURLOPT_POST           => true,
            CURLOPT_POSTFIELDS     => 'grant_type=client_credentials',
            CURLOPT_HTTPHEADER     => ['Accept: application/json']
        ]);

        $raw = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($raw === false) {
            $e = curl_error($ch);
            curl_close($ch);
            throw new Exception($e);
        }

        curl_close($ch);

        $json = json_decode($raw, true);

        if ($status < 200 || $status >= 300) {
            throw new Exception("Token error $status: $raw");
        }

        return $json['access_token'];
    }

    public function create($amount, $currency, $productName, $returnUrl, $cancelUrl)
    {
        $token = $this->token();

        $body = [
            "intent" => "CAPTURE",
            "purchase_units" => [[
                "amount" => [
                    "currency_code" => $currency,
                    "value"         => $amount,
                ]
            ]],
            "application_context" => [
                "return_url" => $returnUrl,
                "cancel_url" => $cancelUrl,
            ]
        ];

        return $this->req('POST', '/v2/checkout/orders', $body, $token);
    }

    public function capture($orderId)
    {
        $token = $this->token();

        $url = $this->base . "/v2/checkout/orders/$orderId/capture";

        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST           => true,
            CURLOPT_HTTPHEADER     => [
                "Content-Type: application/json",
                "Authorization: Bearer $token"
            ]
        ]);

        $raw = curl_exec($ch);
        $status = curl_getinfo($ch, CURLINFO_HTTP_CODE);

        if ($raw === false) {
            $err = curl_error($ch);
            curl_close($ch);
            throw new Exception($err);
        }

        curl_close($ch);

        $json = json_decode($raw, true);

        if ($status < 200 || $status >= 300) {
            throw new Exception("PayPal HTTP $status: $raw");
        }

        return $json;
    }
}
